<?php

namespace App\Http\Controllers;

use App\Http\Requests;
use App\Http\Controllers\Controller;

use App\Service;
use App\Category;
use Illuminate\Http\Request;
use Session;
use App\Classes\USmartApiClass;
use Datatables, Form;

class ServicesController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\View\View
     */
    public function index(Request $request)
    {
        $categoryId = $request['category_id'];
        $price      = $request['price'];
        $status     = $request['status'];
        
        return view('services.index', compact('categoryId', 'price', 'status'));
    }

    public function getData(Request $request)
    {
        $services = Service::where(function ($query) use($request)
            {
                $categoryId = $request['category_id'];
                $price      = $request['price'];
                $status     = $request['status'];

                if($categoryId != '')
                {
                    $query->where('category_id', $categoryId);
                }

                if($price != '')
                {
                    $query->where('price', $price);
                }

                if($status != '')
                {
                    $query->where('services.status', $status);
                }
            });
            
        return Datatables::eloquent($services)
            ->editColumn('category_id', function ($services) {
                return $services->category->name;
            })     
            ->editColumn('price', function ($services) {
                return Form::text('price[]', $services->price, ['class' => 'form-control touchspin']).Form::hidden('id[]', $services->id);
            })    
            ->editColumn('status', function ($services) {
                return setStatusHTML($services->id, $services->status);
            })
            ->addColumn('action', function($services) {
                return '<a href="'.url('/services/'.$services->id.'/edit').'" class="ajaxify"><i class="fa fa-edit"></i></a>';
            })->make(true);
    }
    
    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        return view('services.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     *
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function store(Request $request)
    {
        $this->validate($request, [
            'name'           => 'required|max:255|unique:services',//|alpha_dash_space
            'category_id'    => 'required',
            'price'          => 'required|numeric',
            'process_after'  => 'digits_between:1,7',
            'image'          => 'image|mimes:jpeg,png,jpg,gif|max:200', //KB
            'sort_order'     => 'digits_between:1,7',
            'api_service_id' => 'required_with:api_id|unique:services,api_service_id,NULL,id,api_id,'.$request['api_id']
        ]);
        
        $image = $request->file('image');
        
        if($image != '')
        {
            $fileName = md5(uniqid()).'.'.$image->getClientOriginalExtension();
            $image->move('images/service_images', $fileName);
            
            $request['image_name'] = $fileName;
        }
        
        $request['user_id'] = userId();
        $requestData = $request->except(['brand_id', 'country_id', 'image']);
        
        $service = Service::create($requestData);

        if(!empty($request['brand_id']))
        {
            $service->brands()->sync($request['brand_id']);
        }
        
        if(!empty($request['country_id']))
        {
            $service->countries()->sync($request['country_id']);
        }
        
        Session::flash('flash_message', 'Service added!');

        return response()->json(['response' => 'success']);
    }

    public function autoGeneratedService(Request $request)
    {
        $this->validate($request, [
            'api_id'           => 'required',
            'category_id'      => 'required',
            'api_service_id.*' => 'required'
        ]);
        
        $response = array();
        $serviceId = $request['api_service_id'];
        
        $api = \App\Api::where('id', $request['api_id'])->first();

        if($api->count() > 0)
        {
            $uSmartApiClass = new USmartApiClass($api->api, $api->user_name, $api->url);
            $uSmartApiClass->doAction('imeiservices', array());

            $result = $uSmartApiClass->getResult();
            $uSmartApiClass->XmlToArray($result);
            $arrayData = $uSmartApiClass->createArray();

            if(isset($arrayData['error']) && sizeof($arrayData['error']) > 0)
            {
                return response()->json(['message' => $arrayData['error'][0]]);
            }else
            {
                if(isset($arrayData['Packages']['Package']) && sizeof($arrayData['Packages']['Package']) > 0)
                {
                    $response = $arrayData['Packages']['Package'];
                }

                $total = count($response);
                for($count = 0; $count < $total; $count++)
                {
                    $packageId = $response[$count]['PackageId'];
                    if(in_array($packageId, $serviceId))
                    {
                        $where['name'] = $response[$count]['Category'];
                        $where['type'] = 3;
                        
                        $category = Category::select('id')->where($where)->first();
                        
                        if($category)
                        {
                            $categoryId = $category->id;
                        }else
                        {
                            $data['name'] = trim($response[$count]['Category']);
                            $data['type'] = 3;
                            $data['user_id'] = userId();
                            
                            $categories = Category::create($data);
                            $categoryId = $categories->id;
                        }
                        
                        $request['name']           = trim($response[$count]['PackageTitle']);
                        $request['category_id']    = $categoryId;
                        $request['price']          = $response[$count]['PackagePrice'];
                        $request['cost_price']     = $response[$count]['PackagePrice'];
                        $request['delivery_time']  = $response[$count]['TimeTaken'];
                        $request['terms']          = $response[$count]['MustRead'];
                        $request['api_service_id'] = $packageId;
                        $request['user_id']        = userId();
                        
                        $requestData = $request->all();

                        Service::create($requestData);
                    }    
                }
                
                Session::flash('flash_message', 'Service auto generated!');

                return response()->json(['response' => 'success']);
            }
        }
    }
    
    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     *
     * @return \Illuminate\View\View
     */
    public function edit($id)
    {
        $service = Service::findOrFail($id);

        return view('services.edit', compact('service'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  int  $id
     * @param \Illuminate\Http\Request $request
     *
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function update($id, Request $request)
    {
       $this->validate($request, [
            'name'           => 'required|max:255|unique:services,name,'.$id,//|alpha_dash_space
            'category_id'    => 'required',
            'price'          => 'required|numeric',
            'process_after'  => 'digits_between:1,7',
            'image'          => 'image|mimes:jpeg,png,jpg,gif|max:200', //KB
            'sort_order'     => 'digits_between:1,7',
            'api_service_id' => 'required_with:api_id|unique:services,api_service_id,'.$id.',id,api_id,'.$request['api_id']
        ]);
        
        $image = $request->file('image');
        
        if($image != '')
        {
            if($request['image_name'] == '')
            {
                $fileName = md5(uniqid()).'.'.$image->getClientOriginalExtension();
            }else
            {
                $fileName = $request['image_name'];
            }
            
            $image->move('images/service_images', $fileName);
            
            $request['image_name'] = $fileName;
        }
        
        $request['user_id'] = userId();
        $requestData = $request->except(['brand_id', 'country_id', 'image']);
        
        $service = Service::findOrFail($id);
        $service->update($requestData);

        if(!empty($request['brand_id']))
        {
            $service->brands()->sync($request['brand_id']);
        }
        
        if(!empty($request['country_id']))
        {
            $service->countries()->sync($request['country_id']);
        }
        
        Session::flash('flash_message', 'Service updated!');

        return response()->json(['response' => 'success']);
    }

    public function updatePrice(Request $request)
    {
        $id = $request['id'];
        $price = $request['price'];
        
        foreach ($id as $key => $value):
          
            $data = array(
                'price'   => $price[$key],
                'user_id' => userId(),
            );
            
            Service::where('id', $value)->update($data);
        endforeach;
        
        $message = 'Price(s) updated!';
            
        Session::flash('flash_message', $message);
        return response()->json(['response' => 'success']);
    }
    
    public function updateStatus(Request $request)
    {
        $id = $request['id'];
        $status = $request['status'];
        
        $data['status']  = $status;
        $data['user_id'] = userId();
        
        $result = Service::where('id', $id)->update($data);

        if($result > 0)
        {
            if($status == 1)
            {
                $message = 'Status enabled!';
            }else
            {
                $message = 'Status disabled!';
            }
            
            Session::flash('flash_message', $message);
            return response()->json(['response' => 'success']);
        }else
        {
            $message = 'Status not updated!';
            
            return response()->json(['message' => $message]);
        }
    }
    
    public function getApiCategories($apiId)
    {
        echo getApiCategories($apiId);
    }
    
    public function getApiServices(Request $request)
    {
        $apiId = $request['api_id'];
        $categoryId = $request['category_id'];
        echo getApiServices($apiId, '', $categoryId);
    }
}
