<?php

namespace App\Http\Controllers;

use App\Http\Requests;
use App\Http\Controllers\Controller;

use App\Module;
use App\CustomField;
use App\UserProfile;
use Illuminate\Http\Request;
use Session;

class SettingsController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $title = 'Settings';
        $where['user_id'] = userId();
        
        $modules = Module::where($where)->orWhere('global', 1)->get();
        
        $where['global'] = 0;
        $customFields = CustomField::where($where)->get();
        
        return view('settings.index', compact('title', 'modules', 'customFields'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        $title = 'Settings';
        $subTitle = 'Add Module';
        return view('settings.create', compact('title', 'subTitle'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     *
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function store(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|alpha_dash_space|max:30|unique:modules,name,NULL,id,user_id,'.userId()
        ]);
        
        $request['user_id'] = userId();
        $requestData = $request->all();
        
        Module::create($requestData);

        Session::flash('flash_message', 'Module added!');

        return response()->json(['response' => 'success']);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     *
     * @return \Illuminate\View\View
     */
    public function edit($id)
    {
        $title = 'Settings';
        $subTitle = 'Edit Module';
        $where['user_id'] = userId();
        
        $module = Module::where($where)->findOrFail($id);

        return view('settings.edit', compact('title', 'subTitle', 'module'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  int  $id
     * @param \Illuminate\Http\Request $request
     *
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function update($id, Request $request)
    {
        $this->validate($request, [
            'name' => 'required|alpha_dash_space|max:30|unique:modules,name,'.$id.',id,user_id,'.userId()
        ]);
       
        $requestData = $request->all();
        
        $where['user_id'] = userId();
        $module = Module::where($where)->findOrFail($id);
        $module->update($requestData);

        Session::flash('flash_message', 'Module updated!');

        return response()->json(['response' => 'success']);
    }
    
    public function createCustomField()
    {
        $title = 'Settings';
        $subTitle = 'Add Custom Field';
        return view('settings.create_custom_field', compact('title', 'subTitle'));
    }

    public function storeCustomField(Request $request)
    {
        $this->validate($request, [
            'name'          => 'required|alpha_dash_space|max:30|unique:custom_fields,name,NULL,id,user_id,'.userId(),
            'module_id'     => 'required',
            'field_type_id' => 'required',
            'field_values'  => 'required_if:field_type_id,2',
            'sort_order'    => 'digits_between:1,7'
        ]);
        
        $request['user_id'] = userId();
        $requestData = $request->all();
        
        CustomField::create($requestData);

        Session::flash('flash_message', 'Custom Field added!');

        return response()->json(['response' => 'success']);
    }

    public function editCustomField($id)
    {
        $title = 'Settings';
        $subTitle = 'Edit Custom Field';
        $where['user_id'] = userId();
        
        $customField = CustomField::where($where)->findOrFail($id);

        return view('settings.edit_custom_field', compact('title', 'subTitle', 'customField'));
    }

    public function updateCustomField($id, Request $request)
    {
        $this->validate($request, [
            'name'          => 'required|alpha_dash_space|max:30|unique:custom_fields,name,'.$id.',id,user_id,'.userId(),
            'module_id'     => 'required',
            'field_type_id' => 'required',
            'field_values'  => 'required_if:field_type_id,2',
            'sort_order'    => 'digits_between:1,7'
        ]);
       
        $requestData = $request->all();
        
        $where['user_id'] = userId();
        $module = CustomField::where($where)->findOrFail($id);
        $module->update($requestData);

        Session::flash('flash_message', 'Custom Field updated!');

        return response()->json(['response' => 'success']);
    }
    
    public function updateStatus(Request $request)
    {
        $id = $request['id'];
        $status = $request['status'];
        
        $where['id'] = $id;
        $where['user_id'] = userId();
        
        $result = CustomField::where($where)->update(array('status' => $status));

        if($result > 0)
        {
            if($status == 1)
            {
                $message = 'Status enabled!';
            }else
            {
                $message = 'Status disabled!';
            }
            
            Session::flash('flash_message', $message);
            return response()->json(['response' => 'success']);
        }else
        {
            $message = 'Status not updated!';
            
            return response()->json(['message' => $message]);
        }
    }
    
    public function userProfile(Request $request)
    {
        $page = $request['page'];
        $title = 'My Profile';
        $user_profile = getProfile();

        return view('laravel-authentication-acl::admin.user.user-profile', compact('page', 'title', 'user_profile'));
    }
    
    public function updateUserProfile($id, Request $request)
    {
        if($request['page'] == 1)
        {
           $this->validate($request, [
                'title'       => 'required|alpha|max:5',
                'first_name'  => 'required|alpha_dash_space|max:50',
                'last_name'   => 'required|alpha_dash_space|max:50',
                'phone_no'    => 'digits_between:1,20'
            ]); 
        }else if($request['page'] == 4)
        {
            $this->validate($request, [
                'business_no' => 'digits_between:1,20',
                'company'     => 'required|max:50',
                'vat'         => 'max:20',
                'town'        => 'max:50',
                'county'      => 'max:50',
                'zip'         => 'max:20',
                'address'     => 'required'
            ]);
        }
        
        
        $requestData = $request->all();

        if($request['user_profile'] == 0)
        {
            $where['user_id'] = userId();
            $userProfile = UserProfile::where($where)->findOrFail($id);
        }  else {
            $userProfile = UserProfile::findOrFail($id);
        }
        $userProfile->update($requestData);

        Session::flash('flash_message', 'User updated!');

        return response()->json(['response' => 'success']);
    }
    
    public function updateCompanyProfile($id, Request $request)
    {
        $this->validate($request, [
            'image' => 'image|mimes:jpeg,png,jpg,gif|max:200' //KB
        ]);

        $image = $request->file('image');

        if($image != '')
        {
            if($request['company_logo'] == '')
            {
                $fileName = md5(uniqid()).'.'.$image->getClientOriginalExtension();
            }else
            {
                $fileName = $request['company_logo'];
            }

            $image->move('images/company_profiles', $fileName);

            $request['company_logo'] = $fileName;
        }

        $requestData = $request->all();

        $where['user_id'] = userId();
        $userProfile = UserProfile::where($where)->findOrFail($id);
        $userProfile->update($requestData);
        
        Session::flash('flash_message', 'Updated Invoice Template!');

        return response()->json(['response' => 'success']);
    }
    
    public function updatePageRefreshTime(Request $request)
    {
        $where['user_id'] = userId();

        $data['page_refresh_time'] = $request['page_refresh_time'];
        
        UserProfile::where($where)->update($data);
       
        Session::flash('flash_message', 'Page Refresh Time Updated!');
        return response()->json(['response' => 'success']);
    }
}
